<?php

namespace App\Models;

use App\Core\Database;
use PDO;

class Payment
{
    public int $id;
    public int $user_id;
    public int $package_id;
    public float $amount;
    public string $status;
    public string $provider;
    public ?string $external_id;
    public ?string $redirect_url;
    public string $reference;

    public static function createPending(int $userId, int $packageId, float $amount, string $provider, string $reference): self
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('INSERT INTO payments (user_id, package_id, amount, status, provider, reference) VALUES (:user_id, :package_id, :amount, :status, :provider, :reference)');
        $stmt->execute([
            'user_id' => $userId,
            'package_id' => $packageId,
            'amount' => $amount,
            'status' => 'pending',
            'provider' => $provider,
            'reference' => $reference
        ]);
        $id = (int) $pdo->lastInsertId();
        return self::findById($id);
    }

    public static function attachExternal(int $id, string $externalId, string $redirectUrl): void
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('UPDATE payments SET external_id = :external_id, redirect_url = :redirect_url WHERE id = :id');
        $stmt->execute([
            'external_id' => $externalId,
            'redirect_url' => $redirectUrl,
            'id' => $id
        ]);
    }

    public static function findById(int $id): ?self
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('SELECT * FROM payments WHERE id = :id LIMIT 1');
        $stmt->execute(['id' => $id]);
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$data) {
            return null;
        }
        return self::fromArray($data);
    }

    public static function findByExternalId(string $externalId): ?self
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('SELECT * FROM payments WHERE external_id = :external_id LIMIT 1');
        $stmt->execute(['external_id' => $externalId]);
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$data) {
            return null;
        }
        return self::fromArray($data);
    }

    public static function markPaidByExternalId(string $externalId): ?self
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('SELECT * FROM payments WHERE external_id = :external_id LIMIT 1');
        $stmt->execute(['external_id' => $externalId]);
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$data) {
            return null;
        }
        if ($data['status'] === 'paid') {
            return null;
        }

        $stmt = $pdo->prepare('UPDATE payments SET status = :status WHERE external_id = :external_id');
        $stmt->execute([
            'status' => 'paid',
            'external_id' => $externalId
        ]);

        $data['status'] = 'paid';
        return self::fromArray($data);
    }

    public static function fromArray(array $data): self
    {
        $p = new self();
        $p->id = (int) $data['id'];
        $p->user_id = (int) $data['user_id'];
        $p->package_id = (int) $data['package_id'];
        $p->amount = (float) $data['amount'];
        $p->status = $data['status'];
        $p->provider = $data['provider'];
        $p->external_id = $data['external_id'] ?? null;
        $p->redirect_url = $data['redirect_url'] ?? null;
        $p->reference = $data['reference'];
        return $p;
    }
}

